package mcfall.math;

public class Point extends ColumnVector {
	
	private static final int X_INDEX = 1;
	private static final int Y_INDEX = 2;
	private static final int Z_INDEX = 3;
		
	/**
	 * Constructs a point with the given coordinates
	 * @param x the x-coordinate of the point
	 * @param y the y-coordinate of the point
	 * @param z the z-coordinate of the point
	 */
	public Point (double x, double y, double z) {
		super (4);		
		setX (x);
		setY (y);
		setZ (z);
		setValueAt (Z_INDEX+1, 1);
	}
	
	/**
	 * Constructs a 2-dimensional point with the given coordinates,
	 * by setting the z-coordinate to 0
	 * @param x the x-coordiante of the point
	 * @param y the y-coordinate of the point
	 */
	public Point (double x, double y) {
		this (x, y, 0);
	}

	/**
	 * Retrieve the x-coordinate of this point
	 * @return the x-coordinate of this point
	 */
	public double getX() {
		return getValueAt (X_INDEX);
	}

	/**
	 * Set the x-coordinate of this point, without changing the
	 * other two coordinates
	 * @param x the new x-coordinate for the point
	 */
	public void setX(double x) {
		setValueAt(X_INDEX, x);
	}
	/**
	 * Retrieve the y-coordinate of this point
	 * @return the y-coordinate of this point
	 */
	public double getY() {
		return getValueAt (Y_INDEX);
	}

	/**
	 * Set the y-coordinate of this point, without changing the
	 * other two coordinates
	 * @param y the new y-coordinate for the point
	 */
	public void setY(double y) {
		setValueAt (Y_INDEX, y);
	}

	/**
	 * Retrieve the z-coordinate of this point
	 * @return the z-coordinate of this point
	 */
	public double getZ() {
		return getValueAt (Z_INDEX);
	}

	/**
	 * Set the z-coordinate of this point, without changing the
	 * other two coordinates
	 * @param z the new z-coordinate for the point
	 */
	public void setZ(double z) {
		setValueAt (Z_INDEX, z);
	}
	
	/**
	 * Sets the three components of this point to new values
	 * @param x the new x-coordinate
	 * @param y the new y-coordinate
	 * @param z the new z-coordinate
	 */
	public void set(double x, double y, double z) {
		setX (x);
		setY (y);
		setZ (z);
	}
	
	/**
	 * Computes the Euclidean distance between this point and the
	 * point <i>other</i>
	 * @param other the point to compute the distance to
	 * @return a double value representing the distance between this
	 * point and the point <i>other</i>
	 */
	public double distance (Point other) {
		return Math.sqrt(Math.pow(other.getX()-this.getX(), 2)+Math.pow(other.getY()-this.getY(), 2) + Math.pow(other.getZ()-this.getZ(), 2));	
	}

	/**
	 * Moves this point by the length of <i>v</i> in the direction 
	 * specified by <i>v</i>.  Calling this method is equivalent to
	 * calling moveBy (v, 1)
	 * @param v a Vector specifiying how to move this point
	 */
	public void moveBy (Vector v) {
		//moveBy (v, v.length());
		setX(getX()+v.getValueAt(v.getFirstIndex()));
		setY(getY()+v.getValueAt(v.getFirstIndex()+1));
		if (v.getSize() >= 3) {
			setZ(getZ()+v.getValueAt(v.getFirstIndex()+2));
		}
	}

	/**
	 * Creates a new point which is the result of adding the components of <i>v</i>
	 * to this point's coordinates
	 * @param direction
	 * @return a new Point that is the result of moving from this point's location
	 * along the vector <i>direction</i>
	 */
	public Point add (Vector direction) {
		Point result = new Point (getX(), getY(), getZ());
		result.moveBy(direction);
		return result;
	}

	/**
	 * Creates a Point from the given matrix, which is expected to be 
	 * a column matrix.  This method is a helper when matrix operations
	 * like pre/post multiply return matrices when we want to treat the 
	 * result as a Point
	 * @param matrix
	 * @return a new Point object whose x coordinate is the entry in the first row of <i>matrix</i>,
	 * y coordinate is the entry in the second row of <i>matrix</i>, and z coordinate is the third row
	 * of <i>matrix</i>
	 */
	public static Point fromColumnMatrix(Matrix matrix) {
		if (matrix.getNumberOfColumns() != 1) {
			throw new IncompatibleMatrixException (matrix, null);
		}
		int firstRowIndex = matrix.getFirstRowIndex();
		int firstColumnIndex = matrix.getFirstColumnIndex();
		double x = matrix.getValueAt(firstRowIndex, firstColumnIndex);
		double y = matrix.getValueAt(firstRowIndex+1, firstColumnIndex);
		double z = matrix.getValueAt(firstRowIndex+2, firstColumnIndex);
		return new Point (x, y, z);
	}

	
	@Override
	/**
	 * For Points, the length method returns the length
	 * of a vector from the origin to the coordinates of
	 * this point.
	 * 
	 * This method is overridden by point due to the use
	 * of homogeneous coordinates to represent both Points
	 * and Vectors; if the vector version of length is used,
	 * the extra 1 leads to inaccurate computations.
	 */
	public double length() {
		double x = getX();
		double y = getY();
		double z = getZ();
		return Math.sqrt(x*x + y*y + z*z);
	}

	@Override
	public String toString() {
		return "(" + getX() + ", " + getY() + ", " + getZ () + ")";
	}
	
	@Override
	public boolean equals(Object o) {
		if(o instanceof Point) {
			return this.equals((Point)o);
		}
		return false;
	}
	
	public boolean equals(Point p) {
		return this.getX()==p.getX() && this.getY()==p.getY() && this.getZ()==p.getZ();
	}
}
